<?php

function obtenerEspecificacionesVDE($codigo) {
    $url = VDE_BASE . '/' . slugify($codigo) . '.html';
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 15,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0'
    ]);
    
    $html = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$html) {
        return [];
    }
    
    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    @$dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8'));
    $xpath = new DOMXPath($dom);
    
    $specs = [];
    
    // Extraer especificaciones de divs
    $nodes = $xpath->query('//div[contains(@class, "content-atribute")]');
    foreach ($nodes as $node) {
        $label = $xpath->query('.//div[contains(@class, "label")]', $node);
        $value = $xpath->query('.//div[contains(@class, "value")]', $node);
        
        if ($label->length && $value->length) {
            $key = trim($label->item(0)->textContent, ': ');
            $val = trim($value->item(0)->textContent);
            if ($key && $val) {
                $specs[$key] = $val;
            }
        }
    }
    
    // También buscar en tablas
    if (empty($specs)) {
        $rows = $xpath->query('//table[contains(@class, "additional-attributes")]//tr');
        foreach ($rows as $row) {
            $th = $xpath->query('.//th', $row);
            $td = $xpath->query('.//td', $row);
            
            if ($th->length && $td->length) {
                $key = trim($th->item(0)->textContent, ': ');
                $val = trim($td->item(0)->textContent);
                if ($key && $val) {
                    $specs[$key] = $val;
                }
            }
        }
    }
    
    // Formatear especificaciones al estilo VAELSA
    $resultado = [];
    
    // Orden de prioridad según cotizaciones VAELSA
    $orden = [
        'Acoplamiento NEMA',
        'Número total de etapas',
        'Número de etapas',
        'HP nominal',
        'Potencia',
        'Descarga',
        'Material del cuerpo de la bomba',
        'Material de construcción',
        'Material de la descarga',
        'Fases x voltaje de alimentación',
        'Voltaje',
        'Fases',
        'Amperaje',
        'RPM',
        'Diámetro nominal',
        'Succión',
        'Presión máxima',
        'Caudal',
        'Serie de bomba',
        'Tipo de arranque',
        'Profundidad máxima de inmersión'
    ];
    
    // Agregar especificaciones en orden de prioridad
    foreach ($orden as $campo) {
        foreach ($specs as $key => $val) {
            if (stripos($key, $campo) !== false) {
                $resultado[] = $key . ': ' . $val;
                unset($specs[$key]);
                break;
            }
        }
    }
    
    // Agregar las demás especificaciones que quedaron
    foreach ($specs as $key => $val) {
        if (count($resultado) >= 10) break; // Máximo 10 especificaciones
        $resultado[] = $key . ': ' . $val;
    }
    
    return $resultado;
}

function slugify($str) {
    $str = strtolower(trim($str));
    $str = str_replace(['/', '_', '.', ' '], '-', $str);
    $str = preg_replace('/[^a-z0-9\-]/', '', $str);
    $str = preg_replace('/-+/', '-', $str);
    return trim($str, '-');
}

function obtenerFichaVDE($codigo) {
    return obtenerEspecificacionesVDE($codigo);
}

/**
 * Genera URL de imagen de producto VDE basada en el código
 * Patrón: https://www.vde.com.mx/media/catalog/product/cache/4f45ba8eaaa39c9687d2f12ce1806ac6/[primera]/[segunda]/[codigo_normalizado].(png|jpg)
 * 
 * Ejemplos:
 * - ENERWELL-G5000 -> e/n/enerwell-g5000-a.png
 * - M0.3X05-12 -> m/0/m0.3x05-12.jpg
 * - AQUAPAK XLB20 -> a/q/aquapak-xlb20.png
 * - SIGMA54A -> s/i/sigma54a.jpg
 */
function obtenerImagenVDE($codigo) {
    if (empty($codigo)) return null;
    
    // Normalizar código: minúsculas, espacios a guiones, mantener puntos y guiones
    $normalizado = strtolower(trim($codigo));
    // Mantener letras, números, guiones, puntos y guiones bajos
    $normalizado = preg_replace('/[^a-z0-9\-_\.]/', '-', $normalizado);
    // Reemplazar múltiples guiones consecutivos por uno solo
    $normalizado = preg_replace('/-+/', '-', $normalizado);
    // Limpiar guiones al inicio y final
    $normalizado = trim($normalizado, '-');
    
    if (empty($normalizado)) return null;
    
    // Obtener primera y segunda letra/número
    $primera = substr($normalizado, 0, 1);
    $segunda = substr($normalizado, 1, 1);
    
    // Si no hay segunda letra/número, usar la primera dos veces
    if (empty($segunda) || !preg_match('/[a-z0-9]/', $segunda)) {
        $segunda = $primera;
    }
    
    // Base URL de VDE
    $baseUrl = 'https://www.vde.com.mx/media/catalog/product/cache/4f45ba8eaaa39c9687d2f12ce1806ac6';
    
    // Variantes de nombres de archivo a probar (orden de prioridad según ejemplos)
    $variantes = [
        $normalizado . '-a.png',      // ENERWELL-G5000 -> energwell-g5000-a.png
        $normalizado . '.png',        // AQUAPAK XLB20 -> aquapak-xlb20.png
        $normalizado . '.jpg',        // M0.3X05-12 -> m0.3x05-12.jpg
        $normalizado . '-a.jpg',      // Variante con sufijo -a
    ];
    
    // Retornar la variante más probable (primera)
    // Si no existe, el código de descarga manejará el error
    $url = $baseUrl . '/' . $primera . '/' . $segunda . '/' . $variantes[0];
    
    return $url;
}

/**
 * Verifica si una URL de imagen existe (HEAD request)
 */
function verificarImagenExiste($url) {
    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_NOBODY => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT => 'Mozilla/5.0'
        ]);
        curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return $httpCode >= 200 && $httpCode < 400;
    }
    return false;
}
?>