<?php
require_once LIBS_DIR . 'pdfparser_autoload.php';

use Smalot\PdfParser\Parser;

class VDEPdfParser {
    private string $filepath;
    private Parser $parser;

    public function __construct(string $filepath) {
        $this->filepath = $filepath;
        $this->parser   = new Parser();
    }

    public function extraerProductos(): array {
        $pdf  = $this->parser->parseFile($this->filepath);
        $text = $pdf->getText();

        if (!is_string($text) || trim($text) === '') {
            throw new Exception('No se pudo leer texto del PDF.');
        }

        $text  = str_replace("\r", "\n", $text);
        $lines = array_filter(array_map('trim', explode("\n", $text)), static fn($l) => $l !== '');

        $productos     = [];
        $inTable       = false;
        $current       = null;

        foreach ($lines as $line) {
            // Detecta el encabezado de la tabla
            if (!$inTable && preg_match('/c[o贸]digo\s+art[i铆]culo/i', $line)) {
                $inTable = true;
                continue;
            }
            // Fin de tabla
            if ($inTable && (stripos($line, 'OBSERVACIONES') !== false || stripos($line, 'Subtotal') !== false)) {
                $inTable = false;
                if ($current) { $productos[] = $current; $current = null; }
                continue;
            }
            if (!$inTable) continue;

            // Fila de producto - Formato: "0010 GRD1.5-15-1220A 1 PZA ..."
            if (preg_match('/^\s*(?:\d{3,5})?\s*([A-Z0-9][A-Z0-9\-\/\._]+)\s+(\d+(?:[.,]\d+)?)\s+(PZA\.?|PZAS?|PIEZA|PIEZAS)\b/i', $line, $m)) {
                if ($current) { $productos[] = $current; }

                $codigo = trim($m[1]);
                $codigo = preg_replace('/^\d{3,5}(?=[A-Z])/','', $codigo);

                $unidad = strtoupper(rtrim($m[3], '.'));

                $current = [
                    'codigo'        => $codigo,
                    'cantidad'      => (float)str_replace(',', '.', $m[2]),
                    'unidad'        => $unidad,
                    'descripcion'   => '',
                    'precio_usd'    => 0.0,
                    'precio_mxn'    => 0.0,
                    'precio'        => 0.0,
                    'moneda'        => 'USD', // Por defecto USD, se cambiará si se detecta otra
                    'fecha_entrega' => '',
                    'marca'         => ''
                ];
                continue;
            }

            if (!$current) continue;

            // Precio unitario - Detecta moneda: USD, MXN, MXP, o pesos mexicanos
            if ($current['precio'] === 0.0 && preg_match('/\$\s*([\d\.,]+)\s*(USD|MXN|MXP|PESOS?|MEXICANOS?)?/i', $line, $pm)) {
                $num = str_replace([',', ' '], '', $pm[1]);
                
                // Manejar diferentes formatos de decimales
                if (substr_count($pm[1], ',') === 1 && substr_count($pm[1], '.') > 1) {
                    $num = str_replace('.', '', $pm[1]);
                    $num = str_replace(',', '.', $num);
                } elseif (substr_count($pm[1], ',') === 1 && substr_count($pm[1], '.') === 0) {
                    $num = str_replace(',', '.', $pm[1]);
                }
                
                $precio = (float)$num;
                $current['precio'] = $precio;
                
                // Detectar moneda
                $monedaTexto = isset($pm[2]) ? strtoupper(trim($pm[2])) : '';
                if (in_array($monedaTexto, ['MXN', 'MXP', 'PESO', 'PESOS', 'MEXICANO', 'MEXICANOS']) || 
                    stripos($line, 'peso') !== false || 
                    stripos($line, 'mxn') !== false || 
                    stripos($line, 'mxp') !== false) {
                    $current['moneda'] = 'MXN';
                    $current['precio_mxn'] = $precio;
                    $current['precio_usd'] = 0.0;
                } else {
                    // Por defecto USD si no se especifica
                    $current['moneda'] = 'USD';
                    $current['precio_usd'] = $precio;
                    $current['precio_mxn'] = 0.0;
                }
                continue;
            }

            // Fecha de entrega
            if ($current['fecha_entrega'] === '' && preg_match('/\b(\d{2}[.\-\/]\d{2}[.\-\/]\d{4})\b/', $line, $fm)) {
                $current['fecha_entrega'] = $fm[1];
                continue;
            }

            // Descripci贸n (evitar basura)
            $posiblesBasura = [
                'cedis y tiempo', 'allende', 'monterrey', 'n.l.', 'n.l', 'observaciones', 'vigencia', 'condiciones'
            ];
            $isFecha = (bool)preg_match('/\b\d{2}[.\-\/]\d{2}[.\-\/]\d{4}\b/', $line);
            $isBasura = $isFecha || (bool)preg_match('/' . implode('|', array_map(fn($w)=>preg_quote($w,'/'), $posiblesBasura)) . '/i', $line);

            if (!$isBasura && $current['descripcion'] === '' && mb_strlen($line) > 6 && !preg_match('/^\d+([\.,]\d+)?$/', $line)) {
                $current['descripcion'] = $line;
            }
        }

        if ($current) $productos[] = $current;

        // Fallback
        if (empty($productos)) {
            $productos = $this->fallbackScan($lines);
        }

        return $productos;
    }

    private function fallbackScan(array $lines): array {
        $productos = [];
        $units = '(PZA\.?|PZAS?|PIEZA|PIEZAS)';
        $re = '/^\s*(?:\d{3,5})?\s*([A-Z0-9][A-Z0-9\-\/\._]+)\s+(\d+(?:[.,]\d+)?)\s+' . $units . '\b/i';

        $n = count($lines);
        for ($i=0; $i<$n; $i++) {
            $line = $lines[$i];
            if (!preg_match($re, $line, $m)) continue;

            $codigo = preg_replace('/^\d{3,5}(?=[A-Z])/', '', trim($m[1]));
            $unidad = strtoupper(rtrim($m[3], '.'));
            $cant   = (float)str_replace(',', '.', $m[2]);

            $producto = [
                'codigo'        => $codigo,
                'cantidad'      => $cant,
                'unidad'        => $unidad,
                'descripcion'   => '',
                'precio_usd'    => 0.0,
                'precio_mxn'    => 0.0,
                'precio'        => 0.0,
                'moneda'        => 'USD',
                'fecha_entrega' => '',
                'marca'         => ''
            ];

            // Buscar precio en las siguientes líneas
            for ($j=$i+1; $j<=min($i+6, $n-1); $j++) {
                $ln = $lines[$j];

                if ($producto['precio'] === 0.0 && preg_match('/\$\s*([\d\.,]+)\s*(USD|MXN|MXP|PESOS?|MEXICANOS?)?/i', $ln, $pm)) {
                    $num = str_replace([',', ' '], '', $pm[1]);
                    if (substr_count($pm[1], ',') === 1 && substr_count($pm[1], '.') === 0) {
                        $num = str_replace(',', '.', $pm[1]);
                    }
                    $precio = (float)$num;
                    $producto['precio'] = $precio;
                    
                    // Detectar moneda
                    $monedaTexto = isset($pm[2]) ? strtoupper(trim($pm[2])) : '';
                    if (in_array($monedaTexto, ['MXN', 'MXP', 'PESO', 'PESOS', 'MEXICANO', 'MEXICANOS']) || 
                        stripos($ln, 'peso') !== false || 
                        stripos($ln, 'mxn') !== false || 
                        stripos($ln, 'mxp') !== false) {
                        $producto['moneda'] = 'MXN';
                        $producto['precio_mxn'] = $precio;
                        $producto['precio_usd'] = 0.0;
                    } else {
                        $producto['moneda'] = 'USD';
                        $producto['precio_usd'] = $precio;
                        $producto['precio_mxn'] = 0.0;
                    }
                }

                if ($producto['fecha_entrega']==='' && preg_match('/\b(\d{2}[.\-\/]\d{2}[.\-\/]\d{4})\b/', $ln, $fm)) {
                    $producto['fecha_entrega'] = $fm[1];
                }
            }

            $productos[] = $producto;
        }
        return $productos;
    }
}
?>