<?php
// === Generar Word con PhpWord (DOCX) ===
require_once 'config.php';
require_once LIBS_DIR . 'PHPWord/Autoloader.php';

\PhpOffice\PhpWord\Autoloader::register();

use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\Style\ListItem;
use PhpOffice\PhpWord\SimpleType\Jc;
use PhpOffice\PhpWord\Shared\Converter;

header('Content-Type: application/json; charset=UTF-8');
ini_set('display_errors', '0');
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido'], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    $raw = file_get_contents('php://input');
    if (!$raw) throw new RuntimeException('Payload vacío');
    $data = json_decode($raw, true, 512, JSON_THROW_ON_ERROR);

    // --------- Validación mínima ---------
    $consecutivo   = trim((string)($data['consecutivo'] ?? 'COT-SIN-FOLIO'));
    $fechaTexto    = trim((string)($data['fecha_texto'] ?? ''));
    $clienteNombre = trim((string)($data['cliente_nombre'] ?? ''));
    $clienteEmp    = trim((string)($data['cliente_empresa'] ?? ''));
    $vendedor      = trim((string)($data['vendedor'] ?? ''));
    $mostrarPU     = !empty($data['mostrar_precios_unitarios']);
    $productos     = is_array($data['productos'] ?? null) ? $data['productos'] : [];
    $subtotalFmt   = trim((string)($data['subtotal_fmt'] ?? '$0.00 MXN'));

    if (!$productos) throw new RuntimeException('No hay productos que imprimir.');

    // --------- PhpWord ---------
    $phpWord = new PhpWord();
    // Estilo base
    $phpWord->setDefaultFontName('Arial');
    $phpWord->setDefaultFontSize(10);

    $secStyle = [
        'marginTop'    => Converter::cmToTwip(2),
        'marginBottom' => Converter::cmToTwip(2),
        'marginLeft'   => Converter::cmToTwip(2.5),
        'marginRight'  => Converter::cmToTwip(2.5),
    ];
    $section = $phpWord->addSection($secStyle);

    // ========== Encabezado con logo + fecha y consecutivo ==========
    $headerTable = $section->addTable([
        'borderSize' => 0,
        'alignment' => Jc::CENTER,
        'cellMargin' => 0,
        'unit' => 'pct',
        'width' => 100 * 50
    ]);
    $headerTable->addRow();
    $c1 = $headerTable->addCell(70 * 50);
    $c2 = $headerTable->addCell(30 * 50);

    // Logo (descargar si es URL remota)
    $logoPath = firstExisting([
        BASE_DIR . '/assets/vaelsa.png',
        BASE_DIR . '/assets/logo_vaelsa.png',
        BASE_DIR . '/logo.png'
    ]);
    
    if (!$logoPath && defined('LOGO_URL') && LOGO_URL) {
        $logoPath = downloadImage(LOGO_URL);
    }
    
    if ($logoPath && file_exists($logoPath) && filesize($logoPath) > 0) {
        $c1->addImage($logoPath, ['width'=>210, 'alignment'=>Jc::LEFT, 'wrappingStyle'=>'infront']);
    } else {
        $c1->addText(EMPRESA_NOMBRE, ['bold'=>true, 'size'=>16], ['alignment'=>Jc::LEFT]);
    }
    
    $c2->addText('MONTERREY, N.L. A ' . strtoupper($fechaTexto), ['size'=>10], ['alignment'=>Jc::END, 'spaceAfter'=>0]);
    $c2->addText($consecutivo, ['bold'=>true, 'size'=>10], ['alignment'=>Jc::END]);

    $section->addTextBreak(1);

    // Empresa + ATN (si hay empresa, se muestra arriba y ATN debajo)
    if ($clienteEmp !== '') {
        $section->addText(strtoupper($clienteEmp), ['bold'=>true, 'size'=>12], ['spaceAfter'=>80]);
    }
    $section->addText('ATN.: ' . strtoupper($clienteNombre), ['bold'=>true, 'size'=>12, 'underline'=>'single'], ['spaceAfter'=>160]);

    // Párrafo de introducción
    $intro = 'ATENDIENDO A SU SOLICITUD DE COTIZACIÓN, PONEMOS A SU AMABLE CONSIDERACIÓN EL PRESUPUESTO PARA EL SUMINISTRO DE LO SIGUIENTE:';
    $section->addText($intro, ['bold'=>true, 'size'=>10], ['alignment'=>Jc::BOTH, 'spaceAfter'=>200]);

    // ========= Items =========
    foreach ($productos as $p) {
        $codigo = trim((string)($p['codigo'] ?? ''));
        $marca  = trim((string)($p['marca'] ?? ''));
        $desc   = trim((string)($p['descripcion'] ?? $codigo));
        $cantF  = trim((string)($p['cantidad_fmt'] ?? '1'));
        $unidad = trim((string)($p['unidad'] ?? 'PIEZA'));
        $tit = rtrim($desc.' '.($marca!=='' ? mb_strtoupper(' '.$marca) : '').($codigo!=='' ? ' '.mb_strtoupper($codigo) : ''), ' ');
        $tit .= '. '.$cantF.' '.$unidad.'.';

        // Línea con viñeta (título del producto)
        $section->addListItem(h($tit), 0,
            ['bold'=>true, 'size'=>12],
            ['listType'=>ListItem::TYPE_BULLET_FILLED, 'spaceAfter'=>80]
        );

        // Tabla 2 columnas: specs a la izquierda, imagen opcional a la derecha
        $tbl = $section->addTable([
            'borderSize'=>0,
            'alignment'=>Jc::LEFT,
            'cellMargin'=>50,
            'unit'=>'pct',
            'width'=>100*50
        ]);
        $tbl->addRow();
        $left  = $tbl->addCell(75 * 50);
        $right = $tbl->addCell(25 * 50);

        $specs = is_array($p['especificaciones'] ?? null) ? $p['especificaciones'] : [];
        foreach ($specs as $ln) {
            $left->addText(h($ln), ['size'=>9], ['spaceAfter'=>0, 'lineHeight'=>1.5]);
        }

        if ($mostrarPU) {
            $puFmt = trim((string)($p['precio_unit_fmt'] ?? ''));
            $ptFmt = trim((string)($p['precio_total_fmt'] ?? ''));
            if ($puFmt !== '') $left->addText('Precio unitario : ' . h($puFmt), ['size'=>9], ['spaceAfter'=>0]);
            if ($ptFmt !== '') $left->addText('Precio total : ' . h($ptFmt), ['size'=>9], ['spaceAfter'=>100]);
        }

        // Imagen (si viene url)
        $imgUrl = trim((string)($p['imagen_url'] ?? ''));
        if ($imgUrl !== '') {
            if ($tmp = downloadImage($imgUrl)) {
                $right->addImage($tmp, ['width'=>120, 'alignment'=>Jc::RIGHT]);
                @unlink($tmp);
            }
        }

        $section->addTextBreak(1);
    }

    // Subtotal (siempre visible)
    $section->addText(
        'PRECIO TOTAL EN PESOS MEXICANOS'."\t\t".h($subtotalFmt).' + IVA',
        ['bold'=>true, 'size'=>12],
        ['spaceBefore'=>120, 'spaceAfter'=>240]
    );

    // ========= Pie de página en TODAS las páginas =========
    $footerText = "A T E N T A M E N T E:\n".$vendedor."\nVENTAS\n".EMPRESA_DIRECCION."\nWHATSAPP: ".EMPRESA_WHATSAPP."    mail: ".EMPRESA_EMAIL;
    
    // Agregar footer a la sección actual
    $footer = $section->addFooter();
    $footer->addText($footerText, ['size'=>8], ['alignment'=>Jc::CENTER, 'lineHeight'=>1.0, 'spaceBefore'=>0, 'spaceAfter'=>0]);

    // ========= Condiciones comerciales en la MISMA sección (sin página nueva) =========
    // Agregar salto de página solo si es necesario (dejar que Word lo maneje automáticamente)
    $section->addTextBreak(1);
    $section->addText('CONDICIONES COMERCIALES', ['bold'=>true, 'size'=>12], ['alignment'=>Jc::CENTER, 'spaceAfter'=>200]);

    // Tiempo de entrega basado en los "días adicionales" capturados (rango min–max)
    $dias = array_map(fn($p)=> (int)($p['dias_adicionales'] ?? 0), $productos);
    $dias = array_values(array_filter($dias, fn($x)=> $x>0));
    $te   = $dias ? (min($dias).'–'.max($dias).' días hábiles') : '2–3 días hábiles';

    $condLines = [
        'PRECIOS FIJOS EN PESOS MEXICANOS, MAS IVA (El cual se cargará al facturar).',
        'L.A.B.: MONTERREY, N.L.',
        'TIEMPO DE ENTREGA: '.$te,
        'FORMA DE PAGO: 50% DE ANTICIPO RESTO CONTRA AVISO DE EMBARQUE',
        'VIGENCIA DE COTIZACIÓN: 30 DÍAS',
    ];
    foreach ($condLines as $ln) {
        $section->addText(h($ln), ['size'=>11], ['alignment'=>Jc::LEFT, 'spaceAfter'=>120]);
    }

    // ========= Guardar =========
    // Usar OUTPUT_DIR de config.php
    if (!is_dir(OUTPUT_DIR)) {
        if (!@mkdir(OUTPUT_DIR, 0755, true) && !is_dir(OUTPUT_DIR)) {
            throw new RuntimeException('No se pudo crear la carpeta de salida.');
        }
    }
    if (!is_writable(OUTPUT_DIR)) {
        throw new RuntimeException('La carpeta de salida no es escribible.');
    }

    $filename = preg_replace('~[^A-Za-z0-9._-]+~', '_', $consecutivo) . '.docx';
    $path     = OUTPUT_DIR . $filename;

    $writer = IOFactory::createWriter($phpWord, 'Word2007');
    $writer->save($path);
    
    if (!file_exists($path) || filesize($path) === 0) {
        throw new RuntimeException('El archivo Word no se pudo escribir.');
    }

    // Limpieza opcional de archivos viejos (> 2 días)
    cleanupOld(OUTPUT_DIR, 2);

    // Limpiar logo temporal si se descargó
    if ($logoPath && strpos($logoPath, sys_get_temp_dir()) !== false && file_exists($logoPath)) {
        @unlink($logoPath);
    }

    echo json_encode([
        'ok' => true,
        'filename' => $filename,
        'download_url' => 'output/' . $filename
    ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
}

// ===== Helpers =====
function firstExisting(array $paths): ?string {
    foreach ($paths as $p) {
        if (is_readable($p)) return $p;
    }
    return null;
}

function cleanupOld(string $dir, int $days): void {
    $limit = time() - ($days * 86400);
    foreach (glob($dir . '/*.docx') as $f) {
        if (@filemtime($f) < $limit) @unlink($f);
    }
}

function downloadImage(string $url): ?string {
    // Intentar con cURL primero
    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'Mozilla/5.0',
        ]);
        $bin = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);
        if ($bin !== false && $status >= 200 && $status < 300) {
            $tmp = sys_get_temp_dir() . '/vaelsa_img_' . bin2hex(random_bytes(4)) . '.png';
            if (@file_put_contents($tmp, $bin) !== false) {
                return $tmp;
            }
        }
    }
    
    // Fallback con file_get_contents
    $ctx = stream_context_create(['http'=>['timeout'=>10,'header'=>"User-Agent: Mozilla/5.0\r\n"]]);
    $bin = @file_get_contents($url, false, $ctx);
    if ($bin !== false) {
        $tmp = sys_get_temp_dir() . '/vaelsa_img_' . bin2hex(random_bytes(4)) . '.png';
        if (@file_put_contents($tmp, $bin) !== false) {
            return $tmp;
        }
    }
    
    return null;
}
