// --- Drag & Drop (igual que tienes) ---
const dropZone = document.getElementById('dropZone');
const pdfFile  = document.getElementById('pdfFile');
let fileName   = dropZone.querySelector('.file-name') || (()=>{const d=document.createElement('div');d.className='file-name';dropZone.appendChild(d);return d;})();

['dragenter','dragover','dragleave','drop'].forEach(ev=>{
  dropZone.addEventListener(ev, e=>{ e.preventDefault(); e.stopPropagation(); }, false);
});
dropZone.addEventListener('dragover', ()=> dropZone.classList.add('dragover'));
dropZone.addEventListener('dragleave',()=> dropZone.classList.remove('dragover'));
dropZone.addEventListener('drop',      ()=> dropZone.classList.remove('dragover'));

// ✅ Click robusto: NO hacemos preventDefault/stopPropagation aquí
dropZone.addEventListener('click', (e) => {
  // Si hay <label for="pdfFile"> basta con no hacer nada y dejar que el click nativo llegue
  // Si NO tienes label, forzamos el click del input:
  if (!e.target.closest('input[type="file"]')) {
    pdfFile?.click();
  }
});

dropZone.addEventListener('drop', (e) => {
  const f = e.dataTransfer?.files?.[0];
  if (!f) return;
  if (!/\.pdf$/i.test(f.name)) { alert('Por favor sube un archivo PDF.'); return; }
  const dt = new DataTransfer();
  dt.items.add(f);
  pdfFile.files = dt.files;
  setFileName(f.name);
});

pdfFile.addEventListener('change', (e) => {
  const f = e.target.files?.[0];
  if (f) setFileName(f.name);
});

function setFileName(name){
  fileName.textContent = '📄 ' + name;
  fileName.classList.add('active');
}

// =============== Procesar PDF ===============
document.getElementById('uploadForm').addEventListener('submit', async (e) => {
  e.preventDefault();

  if (!pdfFile.files || pdfFile.files.length === 0) {
    alert('Por favor selecciona un archivo PDF.');
    pdfFile.focus();
    return;
  }

  const formData = new FormData();
  formData.append('pdf', pdfFile.files[0]);

  document.getElementById('loadingPDF').style.display = 'block';
  document.getElementById('btnProcesar').disabled = true;

  try {
    const resp = await fetch('procesar.php', { method: 'POST', body: formData });
    const raw  = await resp.text();
    let json;
    try {
      json = JSON.parse(raw);
    } catch {
      throw new Error(raw.slice(0,300) || 'Respuesta inválida del servidor');
    }

    if (json.error) { alert('Error: ' + json.error); return; }

    // Guardamos productos crudos del proveedor
    window.productosExtraidos = (json.productos || []).map(normalizaProductoProveedor);

    document.getElementById('quotationData').style.display = 'block';
    document.getElementById('productsCard').style.display = 'block';

    renderProductos(window.productosExtraidos);
    applyGlobalMarginUI(); // respetar toggle actual
    document.getElementById('quotationData').scrollIntoView({ behavior: 'smooth' });

  } catch(err) {
    alert('Error al procesar PDF: ' + err.message);
  } finally {
    document.getElementById('loadingPDF').style.display = 'none';
    document.getElementById('btnProcesar').disabled = false;
  }
});

// =============== Margen global ===============
const margenGlobalChk   = document.getElementById('margenGlobal');
const margenGlobalInput = document.getElementById('margenGlobalValor');
const margenHelp        = document.getElementById('margenHelp');

margenGlobalChk.addEventListener('change', applyGlobalMarginUI);
margenGlobalInput.addEventListener('input', () => {
  if (!margenGlobalChk.checked) return;
  const val = toNumber(margenGlobalInput.value);
  document.querySelectorAll('.margen-individual').forEach(inp => { inp.value = isFinite(val)? val : 0; });
});

function applyGlobalMarginUI(){
  const checked = margenGlobalChk.checked;
  if (checked){
    margenGlobalInput.style.display = 'block';
    margenHelp.style.display        = 'block';
    const val = isFinite(toNumber(margenGlobalInput.value))? toNumber(margenGlobalInput.value) : 15;
    document.querySelectorAll('.margen-individual').forEach(inp=>{
      inp.value    = val;
      inp.disabled = true;
      const g = inp.closest('.form-group'); if (g) g.style.display = 'none';
    });
  }else{
    margenGlobalInput.style.display = 'none';
    margenHelp.style.display        = 'none';
    document.querySelectorAll('.margen-individual').forEach(inp=>{
      inp.disabled = false;
      const g = inp.closest('.form-group'); if (g) g.style.display = '';
    });
  }
}

// =============== Render productos (tarjetas) ===============
function renderProductos(productos){
  const container = document.getElementById('productsTable');
  let html = '<div class="products-grid">';
  productos.forEach((p, i)=>{
    // Normalizar moneda para mostrar: MXP también es MXN
    const monedaDisplay = (p.moneda === 'MXP' || p.moneda === 'MXN') ? 'MXN' : (p.moneda || 'USD');
    const precioLbl = monedaDisplay === 'MXN' ? 'Precio MXN' : 'Precio USD';
    const precioVal = formatMoney(p.precio, monedaDisplay);

    html += `
      <div class="product-card">
        <div class="product-header">Producto ${i+1}: ${escapeHtml(p.codigo||'')}</div>

        <div class="product-info">
          <div class="info-item"><div class="info-label">Descripción</div>
            <div class="info-value">${escapeHtml(p.descripcion||'')}</div></div>

          <div class="info-item"><div class="info-label">Cantidad</div>
            <div class="info-value">${escapeHtml(p.cantidad)} ${escapeHtml(p.unidad)}</div></div>

          <div class="info-item"><div class="info-label">${precioLbl}</div>
            <div class="info-value">${precioVal}</div></div>

          <div class="info-item"><div class="info-label">Entrega Proveedor</div>
            <div class="info-value">${escapeHtml(p.fecha_entrega||'Por definir')}</div></div>
        </div>

        <div class="form-grid">
          <div class="form-group">
            <label>Margen % (Ej: 15 para 15%)</label>
            <input type="number" step="0.01" value="15" class="margen-individual form-control" data-index="${i}" required>
          </div>
          <div class="form-group">
            <label>Días adicionales de entrega</label>
            <input type="number" value="3" class="dias-adicionales form-control" data-index="${i}" required>
          </div>
        </div>
      </div>`;
  });
  html += '</div>';
  container.innerHTML = html;
  applyGlobalMarginUI();
}

// =============== Generar cotización (Word) ===============
document.getElementById('generateBtn').addEventListener('click', async () => {
  const form = document.getElementById('quotationForm');
  if (!form.checkValidity()) { form.reportValidity(); return; }

  const fd = new FormData(form);
  const data = {
    consecutivo: fd.get('consecutivo'),
    vendedor: fd.get('vendedor'),
    cliente_nombre: fd.get('cliente_nombre'),
    cliente_empresa: fd.get('cliente_empresa'),
    fecha: fd.get('fecha'),
    fecha_texto: formatearFecha(fd.get('fecha')),
    tipo_cambio: toNumber(fd.get('tipo_cambio')),
    mostrar_precios_unitarios: fd.get('mostrar_precios_unitarios') === 'on',
    productos: []
  };

  const usarGlobal   = margenGlobalChk.checked;
  const margenGlobal = isFinite(toNumber(margenGlobalInput.value)) ? toNumber(margenGlobalInput.value) : 15;

  // Montar productos => SIEMPRE se incluyen (aunque precio sea 0)
  window.productosExtraidos.forEach((prod, idx)=>{
    // Verificar que los inputs existan antes de usarlos
    const margenInput = document.querySelector(`.margen-individual[data-index="${idx}"]`);
    const diasInput = document.querySelector(`.dias-adicionales[data-index="${idx}"]`);
    
    if (!margenInput || !diasInput) {
      console.warn(`Faltan inputs para producto ${idx}:`, prod);
      // Usar valores por defecto si faltan los inputs
    }
    
    const margen = usarGlobal ? margenGlobal : (margenInput ? toNumber(margenInput.value) : 15);
    const diasAd = parseInt((diasInput ? diasInput.value : '0') || '0', 10);

    // precio base a MXN según moneda de origen
    // Si ya viene en MXN/MXP, usar directamente; si viene en USD, convertir
    const baseMXN = (prod.moneda === 'MXN' || prod.moneda === 'MXP')
      ? toNumber(prod.precio)
      : (prod.moneda === 'USD' ? toNumber(prod.precio) * data.tipo_cambio : 0);

    const precioConMargen = (baseMXN>0) ? (baseMXN / (1 - (margen/100))) : 0;
    const totalMXN        = precioConMargen * toNumber(prod.cantidad);

    // SIEMPRE agregar el producto, incluso si falta algún input
    data.productos.push({
      codigo: prod.codigo || '',
      descripcion: prod.descripcion || '',
      marca: prod.marca || '',
      cantidad: toNumber(prod.cantidad) || 1,
      cantidad_fmt: formatearCantidad(prod.cantidad) || '1',
      unidad: pluralizaUnidad(prod.unidad, toNumber(prod.cantidad)),
      precio_unit_fmt: moneyMXN(precioConMargen),
      precio_total_fmt: moneyMXN(totalMXN),
      especificaciones: Array.isArray(prod.especificaciones) ? prod.especificaciones : [],
      dias_adicionales: diasAd,
      imagen_url: prod.imagen_url || '' // opcional
    });
  });

  // Subtotal (solo suma numérica de los fmt)
  const subtotal = data.productos.reduce((acc,p)=> acc + toNumber(fromMoney(p.precio_total_fmt)), 0);
  data.subtotal_fmt = moneyMXN(subtotal);
  
  // Debug: verificar que todos los productos se incluyeron
  console.log(`Total productos extraídos: ${window.productosExtraidos.length}, Productos en data: ${data.productos.length}`);
  if (window.productosExtraidos.length !== data.productos.length) {
    console.warn('⚠️ Diferencia en cantidad de productos!', {
      extraidos: window.productosExtraidos.length,
      enviados: data.productos.length
    });
  }

  document.getElementById('loadingGenerate').style.display = 'block';
  document.getElementById('generateBtn').disabled = true;

  try {
    const resp = await fetch('generar_word.php', {
      method: 'POST',
      headers: { 'Content-Type':'application/json' },
      body: JSON.stringify(data)
    });
    const raw = await resp.text();
    let json; 
    try { json = JSON.parse(raw); } 
    catch { throw new Error(raw.slice(0,500) || 'Respuesta vacía del servidor'); }

    if (json.error) { alert('Error: ' + json.error); return; }

    document.getElementById('resultCard').style.display = 'block';
    document.getElementById('downloadLink').href      = json.download_url;
    document.getElementById('downloadLink').download  = json.filename;
    document.getElementById('resultCard').scrollIntoView({ behavior:'smooth' });

  } catch(err){
    alert('Error al generar documento: ' + err.message);
  } finally {
    document.getElementById('loadingGenerate').style.display = 'none';
    document.getElementById('generateBtn').disabled = false;
  }
});

// =============== Utilidades ===============
function normalizaProductoProveedor(p){
  // Acepta distintos esquemas: {precio_usd}, {precio_mxn}, o {precio, moneda}
  const out = { ...p };
  
  // Normalizar moneda: MXP también es MXN
  const monedaRaw = String(out.moneda || '').toUpperCase();
  const monedaNormalizada = (monedaRaw === 'MXP' || monedaRaw === 'MXN') ? 'MXN' : 'USD';
  
  // Determinar precio y moneda según lo que venga
  if (out.precio_mxn != null && out.precio_mxn > 0) {
    out.precio = toNumber(out.precio_mxn);
    out.moneda = 'MXN';
  } else if (out.precio_usd != null && out.precio_usd > 0) {
    out.precio = toNumber(out.precio_usd);
    out.moneda = 'USD';
  } else if (out.precio != null && out.precio > 0) {
    // Si viene precio directo, usar la moneda que venga (o default USD)
    out.precio = toNumber(out.precio);
    out.moneda = monedaNormalizada === 'MXN' ? 'MXN' : (out.moneda || 'USD');
  } else {
    // Fallback: si no hay precio, usar 0
    out.precio = 0;
    out.moneda = 'USD';
  }

  out.cantidad = toNumber(out.cantidad) || 1;
  out.unidad   = normalizaUnidad(out.unidad || 'PIEZA');
  return out;
}
function normalizaUnidad(u){
  u = String(u||'').trim().toUpperCase();
  const map = { 'PZA':'PIEZA','PZAS':'PIEZAS','PZ':'PIEZA','M':'METRO','MTS':'METRO','MTS.':'METRO' };
  return map[u] || u;
}
function pluralizaUnidad(u, qty){
  u = normalizaUnidad(u);
  const one = Math.abs(qty-1) < 1e-9;
  const dict = {
    'PIEZA': ['PIEZA','PIEZAS'],
    'METRO': ['METRO','METROS'],
    'LITRO': ['LITRO','LITROS'],
    'JUEGO': ['JUEGO','JUEGOS'],
    'SERVICIO': ['SERVICIO','SERVICIOS']
  };
  if (dict[u]) return one ? dict[u][0] : dict[u][1];
  return one ? u : (u.endsWith('S') ? u : u+'S');
}
function formatearFecha(f){
  const meses = ['ENERO','FEBRERO','MARZO','ABRIL','MAYO','JUNIO','JULIO','AGOSTO','SEPTIEMBRE','OCTUBRE','NOVIEMBRE','DICIEMBRE'];
  const d = new Date(f + 'T00:00:00');
  return `${d.getDate()} DE ${meses[d.getMonth()]} DEL ${d.getFullYear()}`;
}
function moneyMXN(n){ return '$' + Number(n||0).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',') + ' MXN'; }
function formatMoney(n, currency){
  const val = Number(n||0).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',');
  return (currency==='USD' ? '$'+val+' USD' : '$'+val+' MXN');
}
function fromMoney(s){ return String(s||'').replace(/[^\d.\-]/g,''); }
function toNumber(x){ const n = Number(String(x??'').toString().replace(/[^\d.\-]/g,'')); return isFinite(n)? n : 0; }
function formatearCantidad(c){ return String(c).replace(/\.0+$/,''); }
function escapeHtml(s){ return String(s||'').replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
